<?php
/**
 * Product Loader & Helper Functions
 * Centralized product handling (JSON-based)
 * 
 * Reads products from:
 * /products/hp-products.json
 * /products/brother-products.json
 * /products/canon-products.json
 * /products/epson-products.json
 *
 * NOTE:
 * - Product IDs MUST be globally unique
 * - This file should be included AFTER config.php
 */

function load_all_products()
{
    static $products = null;

    // Cache products per request
    if ($products !== null) {
        return $products;
    }

    $products = [];

    $basePath = __DIR__ . '/../products/';

    $files = [
        'hp-products.json',
        'brother-products.json',
        'canon-products.json',
        'epson-products.json',
    ];

    foreach ($files as $file) {
        $filePath = $basePath . $file;

        if (!file_exists($filePath)) {
            continue;
        }

        $json = file_get_contents($filePath);
        if (!$json) {
            continue;
        }

        $data = json_decode($json, true);
        if (!is_array($data)) {
            continue;
        }

        foreach ($data as $product) {
            // Ensure required keys exist
            if (!isset($product['id']) || !isset($product['name']) || !isset($product['price'])) {
                continue;
            }

            $products[(int)$product['id']] = $product;
        }
    }

    return $products;
}

/**
 * Get all active products
 */
function get_all_products()
{
    $products = load_all_products();

    return array_filter($products, function ($product) {
        return isset($product['isActive']) && $product['isActive'] === true;
    });
}

/**
 * Get a single product by ID
 */
function get_product_by_id($id)
{
    $products = load_all_products();
    $id = (int)$id;

    return $products[$id] ?? null;
}

/**
 * Check stock availability
 */
function is_product_in_stock($id)
{
    $product = get_product_by_id($id);

    if (!$product) {
        return false;
    }

    return isset($product['stock']) && $product['stock'] > 0;
}
