<?php
require 'includes/config.php';
require 'includes/functions.php';
require 'includes/product-functions.php';

$page_title = 'Shop Printers | ' . SITE_NAME;
$meta_description = 'Browse and shop reliable printers for home and office use.';

cart_init();

/* Load all products */
$products = get_all_products();

/* ----------------------------
   FILTER PARAMETERS
----------------------------- */
$filterBrand    = $_GET['brand'] ?? null;
$filterCategory = $_GET['category'] ?? null;
$searchQuery = isset($_GET['q']) ? trim($_GET['q']) : null;

/* ----------------------------
   APPLY FILTERS
----------------------------- */
$products = array_filter($products, function ($product) use ($filterBrand, $filterCategory, $searchQuery) {

    // Brand filter
    if ($filterBrand && strcasecmp($product['brand'], $filterBrand) !== 0) {
        return false;
    }

    // Category filter
    if ($filterCategory) {
        $categories = $product['categories'] ?? [];
        if (!in_array($filterCategory, $categories)) {
            return false;
        }
    }

    // Search filter
    if ($searchQuery) {
        $haystack = strtolower(
            $product['name'] . ' ' .
            $product['description'] . ' ' .
            $product['brand'] . ' ' .
            ($product['features'] ?? '')
        );

        if (strpos($haystack, strtolower($searchQuery)) === false) {
            return false;
        }
    }

    return true;
});


/* ----------------------------
   ADD TO CART
----------------------------- */
if (isset($_GET['add'])) {
    $pid = (int) $_GET['add'];
    $product = get_product_by_id($pid);

    if ($product && $product['isActive']) {
        cart_add([
            'id'    => $product['id'],
            'name'  => $product['name'],
            'price' => (float)$product['price']
        ]);
        redirect('cart.php');
    }
}

include 'includes/header.php';

/* Helper: trim description */
function trim_words($text, $limit = 15) {
    $words = preg_split('/\s+/', strip_tags($text));
    return count($words) > $limit
        ? implode(' ', array_slice($words, 0, $limit)) . '...'
        : $text;
}
?>

<!-- 🔹 SHOP HERO -->
<section class="shop-hero">
  <div class="container">
    <h1>Shop Printers</h1>
    <p>Browse reliable printers designed for home and office use.</p>
  </div>
</section>

<!-- 🔹 SHOP CONTENT -->
<section class="shop-section">
  <div class="container">

    <div class="row">

      <!-- SIDEBAR FILTER -->
      <div class="col-md-3">
        <div class="shop-sidebar">
          <h3>Filter By Brand</h3>
          <ul>
            <li><a href="shop.php">All Brands</a></li>
            <li><a href="shop.php?brand=HP">HP</a></li>
            <li><a href="shop.php?brand=Brother">Brother</a></li>
            <li><a href="shop.php?brand=Canon">Canon</a></li>
            <li><a href="shop.php?brand=Epson">Epson</a></li>
          </ul>

          <h3>Filter By Category</h3>
          <ul>
            <li><a href="shop.php">All Categories</a></li>
            <li><a href="shop.php?category=home-printer">Home Printer</a></li>
            <li><a href="shop.php?category=office-printer">Office Printer</a></li>
          </ul>
        </div>
      </div>

      <!-- PRODUCTS -->
      <div class="col-md-9">
        <div class="row">

          <?php if (empty($products)): ?>
            <p>No products match your filter.</p>
          <?php endif; ?>

          <?php foreach ($products as $product): ?>

          <div class="col-md-4 product-col">
            <div class="shop-product-card">

              <div class="shop-product-image">
                <a href="product.php?id=<?= $product['id'] ?>">
                  <img src="<?= $product['imageUrl'] ?>" alt="<?= e($product['name']) ?>">
                </a>
              </div>

              <h3>
                <a href="product.php?id=<?= $product['id'] ?>">
                  <?= e($product['name']) ?>
                </a>
              </h3>

              <p class="shop-desc">
                <?= e(trim_words($product['description'], 15)) ?>
              </p>

              <div class="shop-tags">
                <?php
                $tags = !empty($product['features'])
                    ? array_map('trim', explode(',', $product['features']))
                    : [];
                foreach ($tags as $tag):
                ?>
                  <span><?= e($tag) ?></span>
                <?php endforeach; ?>
              </div>

              <div class="shop-bottom">
                <span class="shop-price">
                  <?= price((float)$product['price']) ?>
                </span>

                <a href="shop.php?add=<?= $product['id'] ?>" class="shop-cart-btn">
                  Add to Cart
                </a>
              </div>

            </div>
          </div>

          <?php endforeach; ?>

        </div>
      </div>

    </div>

  </div>
</section>

<?php include 'includes/footer.php'; ?>
